import java.io.*;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

public class Main {

    /**
     * 读取文法文件,返回文法实例
     * @param file 文法文件路径
     * @return 文法实例
     */
    private static Grammar readGrammarDefinition(
            @SuppressWarnings("SameParameterValue") String file) {
        BufferedReader reader = null;
        try {
            reader = new BufferedReader(
                    new InputStreamReader(
                            new FileInputStream(file)
                    )
            );
        } catch (FileNotFoundException e) {
            System.err.println("文法文件" + file + "不存在");
            System.exit(1);
        }
        try {
            String[] V = reader.readLine().split(",");
            String[] T = reader.readLine().split(",");
            String[] P = reader.readLine().split(",");
            String S = reader.readLine();
            return new Grammar(V, T, P, S);
        } catch (IOException e) {
            System.err.println("文法文件" + file + "格式错误");
            System.exit(1);
        }
        return null;
    }

    /**
     * @param grammar 文法实例
     * @return 文法类型
     */
    private static int checkGrammarType(Grammar grammar) {
        StringBuilder variables = new StringBuilder();
        for (String v : grammar.getV()) {
            variables.append(v);
        }
        StringBuilder terminals = new StringBuilder();
        for (String t : grammar.getT()) {
            terminals.append(t);
        }
        Map<String, String> products = new HashMap<>();
        for (String p : grammar.getP()) {
            String [] se = p.split("->");
            products.put(se[0], se[1]);
        }
        String start = grammar.getS();
        if(!isType0(grammar, variables.toString(), terminals.toString(), products, start)) {
            return -1; // 不符合文法定义
        } else if(!isType1(grammar, variables.toString(), terminals.toString(), products, start)) {
            return 0;
        } else if(!isType2(grammar, variables.toString(), terminals.toString(), products, start)) {
            return 1;
        } else if(!isType3(grammar, variables.toString(), terminals.toString(), products, start)) {
            return 2;
        }
        return 3;
    }

    /**
     * @param grammar 文法实例
     * @param variables V的拼接
     * @param terminals T的拼接
     * @param products P的拼接
     * @param start S
     * @return 是3型文法
     */
    private static boolean isType3(Grammar grammar,
                                   String variables,
                                   String terminals,
                                   Map<String, String> products,
                                   String start) {
        boolean leftLinear = true, rightLinear = true;
        // 必须是2型文法
        if (!isType2(grammar, variables, terminals, products, start)) {
            return false;
        }
        // 右线性判断
        for (String b : products.values()) {
            if (!(b.matches("[" + terminals + "][" + variables + "]?")) && !b.equals("")) {
                rightLinear = false;
                break;
            }
        }
        for (String b : products.values()) {
            if (!(b.matches("[" + variables + "]?[" + terminals + "]")) && !b.equals("")) {
                leftLinear = false;
                break;
            }
        }
        return leftLinear || rightLinear;
    }

    /**
     * @param grammar 文法实例
     * @param variables V的拼接
     * @param terminals T的拼接
     * @param products P的拼接
     * @param start S
     * @return 是2型文法
     */
    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private static boolean isType2(Grammar grammar,
                                   String variables,
                                   String terminals,
                                   Map<String, String> products,
                                   String start) {
        // 必须是1型文法
        if (!isType1(grammar, variables, terminals, products, start)) {
            return false;
        }
        // α∈V
        for (String a : products.keySet()) {
            if(!(a.matches("["+variables+"]"))) {
                return false;
            }
        }
        return true;
    }

    /**
     * @param grammar 文法实例
     * @param variables V的拼接
     * @param terminals T的拼接
     * @param products P的拼接
     * @param start S
     * @return 是1型文法
     */
    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private static boolean isType1(Grammar grammar,
                                   String variables,
                                   String terminals,
                                   Map<String, String> products,
                                   String start) {
        // 必须是0型文法
        if (!isType0(grammar, variables, terminals, products, start)) {
            return false;
        }
        // |β|≥|α|
        for(Map.Entry<String, String> kv : products.entrySet()) {
            if(kv.getKey().length() > kv.getValue().length()) {
                return false;
            }
        }
        return true;
    }

    /**
     * @param grammar 文法实例
     * @param variables V的拼接
     * @param terminals T的拼接
     * @param products P的拼接
     * @param start S
     * @return 是0型文法
     */
    @SuppressWarnings("BooleanMethodIsAlwaysInverted")
    private static boolean isType0(Grammar grammar,
                                   String variables,
                                   String terminals,
                                   Map<String, String> products,
                                   String start) {
        // V, T不相交
        if(haveSameEle(grammar.getV(), grammar.getT())) {
            return false;
        }
        // S需要出现在V中
        if (!hasEle(grammar.getV(), start)) {
            return false;
        }
        // α至少要有V的一个出现
        for (String k : products.keySet()) {
            if(getLongestCommonSubstring(variables, k).equals("")) {
                return false;
            }
        }
        // α∈(V∪T)+
        for (String k : products.keySet()) {
            if(!(k.matches("["+variables+terminals+"]+"))) {
                return false;
            }
        }
        // β∈(V∪T)*
        for (String v: products.values()) {
            if(!(v.matches("["+variables+terminals+"]*"))){
                return false;
            }
        }
        return true;
    }

    /**
     * 最长公共子字符串
     * @param str1 字符串1
     * @param str2 字符串2
     * @return 最长公共子字符串
     */
    public static String getLongestCommonSubstring(String str1, String str2){
        int m = str1.length(), n = str2.length(), max = 0, endIndex=-1;
        int [][] dp = new int[m][n];
        for(int i = 0; i < m; i++) {
            for(int j = 0; j < n; j++) {
                if(str1.charAt(i) == str2.charAt(j)) {
                    if(i == 0||j == 0) {
                        dp[i][j]=1;
                    } else {
                        dp[i][j] = dp[i-1][j-1] + 1;
                    }
                    if(max < dp[i][j])
                    {
                        max = dp[i][j];
                        endIndex=i;
                    }
                }
            }
        }
        return str1.substring(endIndex - max + 1,endIndex + 1);
    }

    /**
     * 判断字符串数组是否包含指定字符串
     * @param strArr 字符串数组
     * @param str 指定字符串
     * @return as数组是否包含s
     */
    private static boolean hasEle(String []strArr, String str) {
        for (String a : strArr) {
            if (a.equals(str)) {
                return true;
            }
        }
        return false;
    }

    /**
     * 判断两个字符串数组是否包含相同元素
     * @param strArr1 字符串数组1
     * @param strArr2 字符串数组2
     * @return 是否有相同元素
     */
    private static boolean haveSameEle(String []strArr1, String []strArr2) {
        for (String a : strArr1) {
            for (String b : strArr2) {
                if (a.equals(b)) {
                    return true;
                }
            }
        }
        return false;
    }

    public static void main(String[] args) {
        String file = "inputfiles/Grammar5.txt";
        Grammar grammar = readGrammarDefinition(file);
//        System.out.println(Arrays.toString(grammar.getV()));
//        System.out.println(Arrays.toString(grammar.getT()));
//        System.out.println(Arrays.toString(grammar.getP()));
//        System.out.println(grammar.getS());
        int type = checkGrammarType(grammar);
        switch (type) {
            case -1:
                System.out.println(file + " 不符合文法定义"); break;
            case 0:
                System.out.println(file + " 是短语结构文法"); break;
            case 1:
                System.out.println(file + " 是上下文相关文法"); break;
            case 2:
                System.out.println(file + " 是上下文无关文法"); break;
            case 3:
                System.out.println(file + " 是正规文法"); break;
        }
    }
}

